import os,os.path,sys,shutil,zipfile
from docDBcommon import *

class DocDBmigrateScripts(DocDBcommon):

    DocDBcommon.DOC_TXT = '''
Migration of Dolibarr scripts file in order to have the bypass fille input/ouput function.
This function switch the Dolibarr document files i/o to sql database i/o.
After processing, a data migration is mandatory to keep Dolibarr running.
Use docDBmigrateData.py to do it.
The reverse operation comes back to the default Dolibarr scripts.

pyhton3 docDBmigrateScripts.py [--help] [--dry-run] [--reverse] --dolibarrDir=path_to_htdocs
Anyway, '--help' and '--dry run' prevent any modification.

Examples :
    python3 docDBmigrateScripts.py --dry-run --dolibarrDir=/home/dolibarr/htdocs
        show results without doing nothing
    python3 docDBmigrateScripts.py --dolibarrDir=/home/dolibarr/htdocs
        do the actual migration
    python3 docDBmigrateScripts.py --dry-run --reverse --dolibarrDir=/home/dolibarr/htdocs
        show results reversing migration, without doing nothing
    python3 docDBmigrateScripts.py --reverse --dolibarrDir=/home/dolibarr/htdocs
        reverse migration, restoring scripts at their initial state
'''

    def check(self):
        '''Checking version, configuration, and capabilities to change some scripts'''

        print(MSG_CHECKING % self.dolibarrDir)
        # check Dolibarr directory
        if not os.path.isdir(self.dolibarrDir):
            print(MSG_NOT_A_DIR % self.dolibarrDir)
            print(MSG_SO_LONG)
            sys.exit()
        # look for Dolibarr version
        versionOk = False
        versionsLst = []
        self.dolVersionTxt = ''
        self.dolVersionNumFiles = 0
        self.dolVersionNumChanges = 0
        for dolVersionText,dolVersionNum,lineNumber,dolVersionNumFiles,dolVersionNumChanges in DOL_VERSIONS_INFOS:
            versionFilePath = os.path.join(self.dolibarrDir, 'filefunc.inc.php')
            versionsLst.append(dolVersionNum)
            fileLine = open(versionFilePath,'r').readlines()[lineNumber]
            if dolVersionText in fileLine and dolVersionNum in fileLine:
                versionOk = True
                self.dolVersionTxt = dolVersionNum
                self.dolVersionNumFiles = dolVersionNumFiles
                self.dolVersionNumChanges = dolVersionNumChanges
        versionsTxt = ','.join(versionsLst)
        if not versionOk:
            print(MGS_VERSION_KO % versionsTxt)
            print(MSG_SO_LONG)
            sys.exit()
        print(MSG_VERSION_OK % self.dolVersionTxt)
        # look for conf file
        self.cfgFilePath = os.path.join(self.dolibarrDir, 'conf', 'conf.php')
        print(MSG_CHECK_CFG_FILE % self.cfgFilePath)
        if not os.path.isfile(self.cfgFilePath):
            print(MGS_NO_CFG % self.cfgFilePath)
            print(MSG_SO_LONG)
            sys.exit()
        else:
            self.cfgData = open(self.cfgFilePath,'r').read()
            msgTxt = MSG_CFG_OK % self.cfgFilePath
            if not self.reverse and CFG_DOC_DB_STRING not in self.cfgData:
                msgTxt = MSG_CFG_OK_UPDATE % self.cfgFilePath
            if self.reverse and CFG_DOC_DB_STRING in self.cfgData:
                msgTxt = MSG_CFG_OK_UPDATE % self.cfgFilePath
        print(msgTxt)
        # look for existing documents path definition and existing directory
        docRootPath = ''
        for line in self.cfgData.split('\n'):
            if DOL_ROOT_VAR in line:
                docRootPath = line.split('=')[-1].strip("'; \n")
        print(MSG_CHECK_DOC_PATH % docRootPath)
        if not os.path.isdir(docRootPath):
            if not docRootPath:
                print(MGS_NO_DOL_VAR % (DOL_ROOT_VAR,self.cfgFilePath))
            else:
                print(MGS_NO_DOC_ROOT % docRootPath)
            print(MSG_SO_LONG)
            sys.exit()
        print(MGS_DOC_ROOT_OK % docRootPath)
        # extract some specific scripts
        with zipfile.ZipFile('scriptFiles.zip',mode='r') as archive:
            archive.extractall()
        # look for possible update of some specific scripts
        scriptsOk = True
        self.specificScriptsLst = []
        scriptFilesPath = os.path.join('.','scriptFiles',self.dolVersionTxt,'htdocs')
        print(MSG_SRCS_CHECK % scriptFilesPath)
        for sPath, dirs, fileNames in os.walk(scriptFilesPath):
            for fileName in [x for x in fileNames if x.endswith('.def')]:
                # source paths as registered in docdb/install/scriptFiles
                dbmSrcDefFilePath = os.path.join(sPath, fileName)
                dbmSrcFilePath = os.path.join(sPath, fileName.replace('.def',''))
                # real source file path
                srcFilePath = os.path.join(sPath.replace(scriptFilesPath,self.dolibarrDir), fileName.replace('.def',''))
                # srcFilePath = os.path.join(self.dolibarrDir, fileName.replace('.def',''))
                self.specificScriptsLst.append(srcFilePath)
                #
                dataDbmSrcDefFile = open(dbmSrcDefFilePath,'rb').read()
                dataDbmSrcFile = open(dbmSrcFilePath,'rb').read()
                dataSrcFile = open(srcFilePath,'rb').read()
                if dataSrcFile not in (dataDbmSrcFile,dataDbmSrcDefFile):
                    scriptsOk = False
                    MSG_SRC_KO = 'file %s different from my references, please include it in %s' % (srcFilePath,scriptFilesPath)
                    print(MSG_SRC_KO)
                else:
                    print(dbmSrcFilePath)
        if scriptsOk:
            print(MSG_SRCS_OK % scriptFilesPath)
        else:
            print(MSG_SRCS_KO % scriptFilesPath)
            print(MSG_SO_LONG)
            sys.exit()
        # remove specific scripts
        shutil.rmtree('scriptFiles')

    def start(self):
        '''Do or reverse the migration with a possible simulation'''

        DONT_TOUCH_THESE_FILES_LST = ['filebypass.php',
            'admin.lib.php','barcode.lib.php','Calculation.php','class.soap_server.php','filefunc.inc.php','functions2.lib.php','Html.php',
            'nusoap.php','tcpdfbarcode.modules.php',
            'tcpdf_barcodes_1d.php','tcpdf_barcodes_2d.php',
            'translate.class.php','Migrator.php','mod_syslog_file.php','syslog.php','Xls.php']
        KEYWORDS_FOUND_16 = ('basename', 'chmod', 'clearstatcache', 'closedir', 'copy', 'delete', 'dirname',
            'disk_free_space', 'disk_total_space', 'exif_read_data', 'fclose', 'feof', 'fflush',
            'fgetc', 'fgetcsv', 'fgets', 'file', 'file_exists', 'file_get_contents', 'file_put_contents',
            'filectime', 'fileinode', 'filemtime', 'fileperms', 'filesize', 'flock', 'fnmatch', 'fopen',
            'fputcsv', 'fputs', 'fread', 'fseek', 'ftell', 'ftruncate', 'fwrite', 'getimagesize', 'glob',
            'imagecreatefromgif', 'imagecreatefromjpeg', 'imagecreatefrompng', 'imagecreatefromwbmp',
            'imagecreatefromwebp', 'imagegif', 'imagejpeg', 'imagepng', 'imagewbmp', 'is_dir', 'is_file',
            'is_link', 'is_readable', 'is_writable', 'is_writeable', 'md5_file', 'mkdir', 'move_uploaded_file',
            'opendir', 'readdir', 'readfile', 'realpath', 'rename', 'rewind', 'rewinddir', 'rmdir',
            'scandir', 'symlink', 'tempnam', 'tmpfile', 'touch', 'unlink')
        KEYWORDS_NO_CONCERN = ('basename','copy','delete','dirname','fnmatch','fpasstglob(hru','fscanf','fpassthru',
            'fstat','is_uploaded_file','link','pathinfo','pclose','popen','umask')
        PREV_CHARS_LIST = ' /\t@=!(,'
        FUNCTION_KEYWORD = 'function'

        # loop on dolibarr htdocs files
        print(MSG_STARTING % self.dolibarrDir)
        nbFilesMigrated = 0
        nbChanged = 0
        for sPath, dirs, fileNames in os.walk(self.dolibarrDir):
            if DOC_DB_DIR in sPath:
                continue
            # some php scripts only
            for fileName in [x for x in fileNames if x.endswith('.php') and x not in DONT_TOUCH_THESE_FILES_LST]:
                srcFilePath = os.path.join(sPath,fileName)
                if srcFilePath in self.specificScriptsLst:
                    continue
                radicPath = srcFilePath.replace(self.dolibarrDir, '..')
                srcData = open(srcFilePath, 'r').read()
                keywordsChangedSet = set()
                if not self.reverse:
                    # normal migration case, we do it
                    for keyword in [x for x in KEYWORDS_FOUND_16 if x not in KEYWORDS_NO_CONCERN]:
                        posStart = 0
                        while True:
                            posFound = srcData[posStart:].find(keyword)
                            if posFound == -1:
                                break
                            posStart += posFound
                            posEnd = posStart + len(keyword)
                            prevChar = srcData[posStart - 1: posStart]
                            functionKeyWordStart = posStart - 1 - len(FUNCTION_KEYWORD)
                            prevFunctionDef = srcData[functionKeyWordStart : functionKeyWordStart + len(FUNCTION_KEYWORD)]
                            if prevChar in PREV_CHARS_LIST and srcData[posEnd] == '(' and prevFunctionDef != FUNCTION_KEYWORD:
                                doIt = True
                                # specific case of file_exists(" ... /main.inc.php")
                                if keyword == 'file_exists':
                                    # finding out expression between parenthesis
                                    wps = posEnd + 1
                                    wpe = wps
                                    while True:
                                        if srcData[wpe] == ')':
                                            break
                                        wpe += 1
                                    expr = srcData[wps:wpe]
                                    # won't do it because bypass functions not yet loaded at php exec time
                                    if 'main.inc.php' in expr:
                                        doIt = False
                                if doIt:
                                    rplTxt = keyword + '_Bypass'
                                    if prevChar == '@':
                                        posStart -= 1
                                        rplTxt += 'A'
                                    srcData = srcData[:posStart] + rplTxt + srcData[posEnd:]
                                    keywordsChangedSet.update({keyword})
                                    nbChanged += 1
                            posStart = posEnd
                else:
                    # reverse migration case, reset to initial state
                    BYPASS_KEY = '_Bypass'
                    BYPASS_A_KEY = '_BypassA'
                    for keyword in [x for x in (BYPASS_A_KEY,BYPASS_KEY)]:
                        posStart = 0
                        while True:
                            posFound = srcData[posStart:].find(keyword)
                            if posFound == -1:
                                break
                            posStart += posFound
                            # look for previous char position
                            for posPrevChar in range(posStart,0,-1):
                                if srcData[posPrevChar] in PREV_CHARS_LIST:
                                    wChar = ''
                                    if keyword == BYPASS_A_KEY:
                                        wChar = '@'
                                    rplTxt = wChar + srcData[posPrevChar + 1:posStart]
                                    srcData = srcData[:posPrevChar + 1] + rplTxt + srcData[posStart + len(wChar) + len(BYPASS_A_KEY) - 1:]
                                    keywordsChangedSet.update({keyword})
                                    nbChanged += 1
                                    posStart = posPrevChar + 1 + len(rplTxt)
                                    break
                if len(keywordsChangedSet) > 0:
                    keywordsChangedLst = list(keywordsChangedSet)
                    keywordsChangedLst.sort()
                    print('%s %s' % (radicPath,keywordsChangedLst))
                    if not self.dryRun:
                        srcFile = open(srcFilePath, 'w')
                        srcFile.write(srcData)
                        srcFile.close()
                    nbFilesMigrated += 1
        # config file
        cfgData = self.cfgData
        if self.reverse and CFG_DOC_DB_STRING in self.cfgData:
            cfgData = ''
            for cfgLine in self.cfgData.split('\n'):
                if CFG_DOC_DB_STRING not in cfgLine:
                    cfgData = cfgData + cfgLine + '\n'
        elif not self.reverse and CFG_DOC_DB_STRING not in self.cfgData:
            cfgData = cfgData + '\n' + CFG_DOC_DB_STRING
        if cfgData != self.cfgData:
            print(MSG_UPDATE_CFG_FILE)
            if not self.dryRun:
                cfgData = cfgData.strip()
                cfgDataFile = open(self.cfgFilePath,'w')
                cfgDataFile.write(cfgData)
                cfgDataFile.close()
        # extract some specific scripts
        with zipfile.ZipFile('scriptFiles.zip',mode='r') as archive:
            archive.extractall()
        # specific sources files
        scriptFilesPath = os.path.join('.', 'scriptFiles', self.dolVersionTxt, 'htdocs')
        print(MSG_SRCS_UPDATE % scriptFilesPath)
        for sPath, dirs, fileNames in os.walk(scriptFilesPath):
            for fileName in [x for x in fileNames if x.endswith('.def')]:
                # source paths as registered in docDBmigrate/scriptFiles
                dbmSrcDefFilePath = os.path.join(sPath, fileName)
                dbmSrcFilePath = os.path.join(sPath, fileName.replace('.def',''))
                # real source file path
                srcFilePath = os.path.join(sPath.replace(scriptFilesPath,self.dolibarrDir), fileName.replace('.def',''))
                #
                dataDbmSrcDefFile = open(dbmSrcDefFilePath,'rb').read()
                dataDbmSrcFile = open(dbmSrcFilePath,'rb').read()
                if not self.dryRun:
                    srcFile = open(srcFilePath, 'wb')
                    if not self.reverse:
                        # write the updated version of the file
                        srcFile.write(dataDbmSrcFile)
                    else:
                        # write the default version
                        srcFile.write(dataDbmSrcDefFile)
                    srcFile.close()
                print(srcFilePath)
        #
        filebypassSrcPath = os.path.join('.', 'scriptFiles', self.dolVersionTxt, 'htdocs', FILE_BYPASS_PHP)
        filebypassDestPath = os.path.join(self.dolibarrDir, FILE_BYPASS_PHP)
        if not self.reverse:
            print(MSG_COPY_BYPASS_FILE)
            if not self.dryRun:
                shutil.copy(filebypassSrcPath,filebypassDestPath)
        else:
            if os.path.exists(filebypassDestPath):
                print(MSG_REMOVE_BYPASS_FILE)
                if not self.dryRun:
                    os.remove(filebypassDestPath)
        # remove specific scripts
        shutil.rmtree('scriptFiles')
        #
        if nbFilesMigrated == self.dolVersionNumFiles and nbChanged == self.dolVersionNumChanges:
            if self.dryRun:
                msg = MSG_FINISHED_OK_DRY
            else:
                msg = MSG_FINISHED_OK
            print(msg % (nbFilesMigrated,nbChanged))
        else:
            if self.dryRun:
                msg = MSG_FINISHED_KO_DRY
            else:
                msg = MSG_FINISHED_KO
            print(msg % (nbFilesMigrated,nbChanged,self.dolVersionNumFiles,self.dolVersionNumChanges))

try:
    obj = DocDBmigrateScripts()
    obj.init()
    obj.check()
    obj.confirm()
    obj.start()
except KeyboardInterrupt:
    print(MSG_INTERRUPTED)