import os,os.path,sys,subprocess,datetime
from docDBcommon import *

class DocDBmigrateFiles(DocDBcommon):

    DocDBcommon.DOC_TXT = '''
Migration of Dolibarr documents file to database.
This function loads the Dolibarr document files into the sql database.
Before processing, a script files migration is mandatory.
Use docDBmigrateScripts.py to do it.
The reverse operation unload the sql database Dolibarr documents into the document directory.

pyhton3 docDBmigrateFiles.py [--help] [--dry-run] [--list-doc] [--reverse] --dolibarrDir=path_to_htdocs
Anyway, '--help' and '--dry run' prevent any modification.

Examples :
    python3 docDBmigrateFiles.py --dry-run --dolibarrDir=/home/dolibarr/htdocs
        show results without doing nothing
    python3 docDBmigrateFiles.py --dolibarrDir=/home/dolibarr/htdocs
        do the actual migration
    python3 docDBmigrateFiles.py --dry-run --reverse --dolibarrDir=/home/dolibarr/htdocs
        show results reversing migration, without doing nothing
    python3 docDBmigrateFiles.py --reverse --dolibarrDir=/home/dolibarr/htdocs
        reverse migration, restoring scripts at their initial state
'''
    try:
        import mysql.connector
    except ModuleNotFoundError:
        print(MSG_INSTALL_MYSQL)
        print(MSG_SO_LONG)
        sys.exit()

    def check(self):
        '''Checking configuration'''

        print(MSG_CHECKING % self.dolibarrDir)

        # check Dolibarr directory
        if not os.path.isdir(self.dolibarrDir):
            print(MSG_NOT_A_DIR % self.dolibarrDir)
            print(MSG_SO_LONG)
            sys.exit()

        # look for conf file
        self.cfgFilePath = os.path.join(self.dolibarrDir, 'conf', 'conf.php')
        print(MSG_CHECK_CFG_FILE % self.cfgFilePath)
        if not os.path.isfile(self.cfgFilePath):
            print(MGS_NO_CFG % self.cfgFilePath)
            print(MSG_SO_LONG)
            sys.exit()
        else:
            self.cfgData = open(self.cfgFilePath,'r').read()
            msgTxt = MSG_CFG_OK % self.cfgFilePath
        print(msgTxt)

        # look for existing documents path definition and existing directory, and also mysql stuff
        self.dbVarsDic = {'dolibarr_main_db_host':'localhost',
            'dolibarr_main_db_port':'3306',
            'dolibarr_main_db_name':'',
            'dolibarr_main_db_prefix':'llx_',
            'dolibarr_main_db_user':'',
            'dolibarr_main_db_pass':'',
            'dolibarr_main_db_type':'mysqli' }
        self.docRootPath = ''
        for line in self.cfgData.split('\n'):
            if DOL_ROOT_VAR in line:
                self.docRootPath = line.split('=')[-1].strip("'; \n")
            for dbVar in self.dbVarsDic:
                if '$'+dbVar in line:
                    self.dbVarsDic[dbVar] = line.split('=')[-1].strip("'; \n")
        print(MSG_CHECK_DOC_PATH % self.docRootPath)
        if not os.path.isdir(self.docRootPath):
            if not self.docRootPath:
                print(MGS_NO_DOL_VAR % (DOL_ROOT_VAR,self.cfgFilePath))
            else:
                print(MGS_NO_DOC_ROOT % self.docRootPath)
            print(MSG_SO_LONG)
            sys.exit()
        print(MGS_DOC_ROOT_OK_FOR_READ % self.docRootPath)

        # check database connection
        print(MSG_CHECK_DB_CONNECTION % self.dbVarsDic)
        dbType = self.dbVarsDic['dolibarr_main_db_type']
        if dbType != 'mysqli':
            print(MSG_ONLY_MYSQL % dbType)
            print(MSG_SO_LONG)
            sys.exit()
        self.db = DocDBmigrateFiles.mysql.connector.connect(
            host = self.dbVarsDic['dolibarr_main_db_host'],
            port = self.dbVarsDic['dolibarr_main_db_port'],
            user = self.dbVarsDic['dolibarr_main_db_user'],
            password = self.dbVarsDic['dolibarr_main_db_pass'],
            database = self.dbVarsDic['dolibarr_main_db_name'])

        # read documents information
        print(MSG_CONNECTED % self.dbVarsDic)
        self.docsLst = set()
        cur = self.db.cursor()
        if not self.reverse:
            # building documents list
            sql = '''(select 'mycompany/logos' as filepath, value as filename 
                from %(dolibarr_main_db_prefix)sconst
                where name in ('MAIN_INFO_SOCIETE_LOGO','MAIN_INFO_SOCIETE_LOGO_SQUARRED') 
                union select filepath,filename 
                    from %(dolibarr_main_db_prefix)secm_files) 
                order by filepath''' % self.dbVarsDic
            cur.execute(sql)
            # fetching db table
            for filepath,filename in cur.fetchall():
                documentPath = os.path.join(self.docRootPath, filepath, filename)
                if os.path.isfile(documentPath):
                    self.docsLst.add((filepath,filename))
                    thumbsPath = os.path.join(self.docRootPath,filepath,'thumbs')
                    if os.path.isdir(thumbsPath):
                        for thumbName in os.listdir(thumbsPath):
                            wFilePath = os.path.join(filepath, 'thumbs')
                            self.docsLst.add((wFilePath,thumbName))
                else:
                    print(MGS_INVALID_FILE % documentPath)
            # walking documents directory (photos are not in database)
            print(MSG_READING_DIR)
            for sPath, dirs, fileNames in os.walk(self.docRootPath):
                for fileName in fileNames:
                    wPath = sPath.replace(self.docRootPath,'').lstrip(os.sep).lstrip('/')
                    tupleDoc = wPath,fileName
                    if wPath and not fileName.endswith('.cache') and tupleDoc not in self.docsLst:
                        self.docsLst.add(tupleDoc)
        else:
            # storing documents list if reverse migration
            sql = '''select path_name from %(dolibarr_main_db_prefix)sdoc_data 
                order by path_name''' % self.dbVarsDic
            cur.execute(sql)
            for result in cur.fetchall():
                path_name = result[0]
                self.docsLst.add(path_name)
        cur.close()
        if self.listDoc:
            for docPath in sorted(self.docsLst):
                w1,w2 = docPath
                print('',os.path.join(w1,w2))
        print(MSG_INFO_DOCS % len(self.docsLst))

    def start(self):
        if not self.reverse:
            self.startMigrate()
        else:
            self.startReverse()

    def startMigrate(self):
        '''Do the migration with a possible simulation'''

        print(MSG_CREATE_DOCUMENTS)
        cur = self.db.cursor()
        # tables creation
        if not self.dryRun:
            for sql in (
                # charset and collate option are needed to emulate file system behaviour where É is different from E
                '''create table llx_doc_directory (
                    rowid bigint(20) not null auto_increment,
                    path_name varchar(1024) default null,
                    primary key (rowid),
                    index (path_name)) default charset=utf8 collate=utf8_bin''',
                '''create table llx_doc_data (
                    rowid bigint(20) not null auto_increment,
                    path_name varchar(1024) default null,
                    filemtime datetime default null,
                    datablob longblob default null,
                    exif_data blob default null,
                    imagesize varchar(1024) default null,
                    primary key (rowid),
                    index (path_name)) default charset=utf8 collate=utf8_bin'''):
                try:
                    cur.execute(sql)
                except DocDBmigrateFiles.mysql.connector.Error as err:
                    if 'already exists' not in str(err):
                        print("something went wrong: {}".format(err))
                        print(MSG_SO_LONG)
                        sys.exit()
        # documents loop
        dirSet = set()
        for filepath,filename in sorted(self.docsLst):
            documentPath = os.path.join(self.docRootPath,filepath,filename)
            wpath = os.path.join(os.sep,filepath,filename)
            wpath = wpath.replace('\\','/')
            print('',wpath)
            # get file data (use php because python converters could give different results)
            phpDataTplPath = os.path.join('templates','getPhpDocData.template')
            filemtimePath = os.path.join('tmp','getPhpDocData.filemtime.tmp')
            imagesizePath = os.path.join('tmp','getPhpDocData.imagesize.tmp')
            exif_dataPath = os.path.join('tmp','getPhpDocData.exif_data.tmp')
            getPhpDataPhpFilePath = os.path.join('tmp','getPhpDocData.php')
            getPhpData = open(phpDataTplPath,'r').read()
            documentPathEscaped = documentPath.replace("'","\\'")
            getPhpData = getPhpData % (documentPathEscaped,filemtimePath,imagesizePath,exif_dataPath)
            tmpScriptFile = open(getPhpDataPhpFilePath,encoding='utf-8',mode='w')
            tmpScriptFile.write(getPhpData.strip())
            tmpScriptFile.close()
            if not self.phpDir:
                cmdLine = 'php %s' % getPhpDataPhpFilePath
            else:
                cmdLine = os.path.join(self.phpDir,'php %s' % getPhpDataPhpFilePath)
            process = subprocess.Popen(cmdLine, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            result,err = process.communicate()
            os.unlink(getPhpDataPhpFilePath)
            filemtime = datetime.datetime.fromtimestamp(int(open(filemtimePath,encoding='utf-8',mode='r').read()))
            imagesize = open(imagesizePath,encoding='utf-8',mode='r').read()
            exif_data = open(exif_dataPath,mode='rb').read()
            os.unlink(filemtimePath)
            os.unlink(imagesizePath)
            os.unlink(exif_dataPath)
            datablob = open(documentPath,'rb').read()
            # write document in database
            if not self.dryRun:
                sql = '''delete from %(dolibarr_main_db_prefix)sdoc_data ''' % self.dbVarsDic + \
                    '''where path_name=%s'''
                values = (wpath,)
                cur.execute(sql,values)
                sql = '''insert into %(dolibarr_main_db_prefix)sdoc_data ''' % self.dbVarsDic + \
                      '''(path_name,filemtime,exif_data,imagesize,datablob) values (%s,%s,%s,%s,%s)'''
                values = (wpath, filemtime, exif_data, imagesize, datablob)
                cur.execute(sql, values)
            # keep all names of directory branch
            while wpath:
                newpath = os.path.split(wpath)[0]
                wpath = newpath
                if wpath == '/':
                    wpath = ''
                dirSet.add(wpath)
        # directories loop
        if not self.dryRun:
            print(MSG_CREATE_DIRECTORIES)
            for wpath in sorted(dirSet):
                if not self.dryRun:
                    sql = '''delete from %(dolibarr_main_db_prefix)sdoc_directory ''' % self.dbVarsDic + \
                        '''where path_name=%s'''
                    values = (wpath,)
                    cur.execute(sql,values)
                    sql = '''insert into %(dolibarr_main_db_prefix)sdoc_directory ''' % self.dbVarsDic + \
                          '''(path_name) values (%s)'''
                    values = (wpath,)
                    cur.execute(sql, values)
        #
        if self.dryRun:
            print(MSG_FINISHED_DATA_OK_DRY)
        else:
            cur.execute('commit')
            print(MSG_FINISHED_DATA_OK % self.docRootPath)
        cur.close()

    def startReverse(self):
        '''Reverse the migration with a possible simulation'''

        print(MSG_CREATE_DOCUMENTS)
        cur = self.db.cursor()
        # documents loop
        dirSet = set()
        for path_name in sorted(self.docsLst):
            wpath_name = path_name.lstrip('/')
            documentPath = os.path.join(self.docRootPath,wpath_name)
            documentPath = documentPath.replace('/',os.sep)
            print(documentPath)
            # read document mtime and data
            sql = '''select filemtime,datablob from %(dolibarr_main_db_prefix)sdoc_data ''' % self.dbVarsDic \
                + '''where path_name=%s '''
            data = (path_name,)
            cur.execute(sql,data)
            filemtime,datablob = cur.fetchone()
            # documentPath = os.path.join(self.docRootPath, documentPath.lstrip(os.sep))
            # create directory if missing
            wpath = os.path.split(documentPath)[0]
            if not self.dryRun:
                os.makedirs(wpath,exist_ok=True)
            # write document on file system
            if not self.dryRun:
                docFile = open(documentPath,mode='wb')
                docFile.write(datablob)
                docFile.close()
                # reflect creation time
                os.utime(documentPath,times=(filemtime.timestamp(),filemtime.timestamp()))
        #
        if self.dryRun:
            print(MSG_FINISHED_DATA_REVERSE_OK_DRY)
        else:
            print(MSG_FINISHED_DATA_REVERSE_OK)
            self.dbVarsDic['docRootPath'] = self.docRootPath
            print(MSG_ADVISE % self.dbVarsDic)
        cur.close()

try:
    obj = DocDBmigrateFiles()
    obj.init()
    obj.check()
    obj.confirm()
    obj.start()
except KeyboardInterrupt:
    print(MSG_INTERRUPTED)
